#ifndef SEARCHER_H
#define SEARCHER_H

#include "defs.h"
#include "board.h"
#include "history.h"
#include "material.h"
#include "movelist.h"
#include "clock.h"
#include "hashtab.h"
#include "eval.h"
#include "pawntab.h"
#include "book.h"

//predefinition of classes inlcuded in search class
class cBoard;
class cMaterial;
class cMovelist;
class cHistory;
class cClock;
class cHashtable;
class cPawntable;
class cEval;
class cBookread;

//search modes
const uint smPERFTSINGLE = 1;
const uint smPERFTFILE = 2;
const uint smDEPTHSET = 4;
const uint smPONDER = 8;
const uint smINFINITE = 16;
const uint smNONE = 32;
const uint STOPPED = 1;
const uint RUNNING = 2;


struct sData {
       u64 nodes;
       u64 qnodes;
	   uint bestmove;
	   uint pondermove;
	   uint pvlen[maxply];
	   uint pv[maxply][maxply];
	   bool incheck[maxply];
	   char commandin[0xffff];
	   uint currentdepth;
	   int lastscore;
};

struct sParameters {
       uint iterdepth;
       uint nullred;
       bool ponderhit; //set to true by protocol criver if we're entering the search with a ponderhit
       bool special;
};

struct sStatus {
       uint mode;
       uint stopped;
       bool interrupted;
};

struct sStats {
    int nulltry;
    int nullcuts;
    int hashcuts;
    int fh;
    int fhf;
    int checkext;
    int pawnseventh;
    int intopawnend;
    int pvssearch;
    int pvsresearch;
};

struct sSearchOpt {
    bool usehash;
    bool donull;
    bool pvs;
    bool special; //used for interrupt in epd or benchmark searches
    int nullredfact;
    string inifilename;
    bool aspiration;
    int aspwindow;
};


class cSearcher {

private:

        sData data[1];
        sParameters param[1];
        sStatus status[1];
        sStats stats[1];

        cSearcher( const cSearcher & );
        cSearcher &operator = ( const cSearcher & searcher);

        void go(uint depth);
        void goroot(uint depth);
        void perftsingle();
        void perftfile();
        void perftnormal(uint depth);
		void iter_deep();
		int  presearch(const uint depth);
		int  alphabeta(int alpha, int beta, int depth, bool nullperm, bool check);
		int  quies(int alpha, int beta);
		void updateiteration(const int score, const uint d);
		void printpv();
		void picknext(const int from, cMovelist &mlist);
		void pvupdate(const uint move);
		void loopcleanup(const int score);
		bool checkdraw();
		void printstats();
		void resetstats();
		void extender(int pvnode, uint &ext, uint move);

		void processini(vector<string> options);
		uint movefromhash(uint makemefirst);
		int checkinput();

public:

        cSearcher();

        sSearchOpt opt[1];
        void init_opt();

       //accessors
       u64 getallnodes() {return data->nodes+data->qnodes;}
       u64 getqnodes() {return data->qnodes;}
       u64 getnodes() {return data->nodes;}
       uint &getstartdepth() {return param->iterdepth;}
       sStats *getstats() { return stats; }
       sData *getdata() { return data; }

       // modifiers
       void incrnodes() { data->nodes++; }
       void incrqnodes() { data->qnodes++; }
	   void setbestmove(const uint b);

       //setters
       void setponderhit(bool set) {param->ponderhit = set;}
       void resetdata();
       void resetparam();
       void setiterdepth(const uint depth) { param->iterdepth = depth; }
       void setmode(const uint mode) { status->mode = mode; }
       void setnullred(const uint nr) { param->nullred = nr; }
       void setininame(string name);

       //main "run" function - called to start the engine searching, perfting etc.
       void compute();
       bool checkmoveislegal(uint move);

       char *hascommand() { return data->commandin; }
       uint getpondermove() { return data->pondermove; }
	   uint getbestmove() { return data->bestmove; }
	   uint getmode() {return status->mode;}
       bool interrupted() { return status->interrupted; }
       void resetinterrupted() { status->interrupted = false; }

	   void readini(string file);
	   void showinivals();

       //classes that are part of the search object
       cMovelist mlist;
       cBoard game;
       cMaterial mat;
       cHistory his;
       cClock timer;
       cHashtable ttable;
       cPawntable ptab;
       cEval scorer;
       cBookread book;
       bool uci;
       bool xb;

};

extern int checkinput();

#endif
